var gdl90 = {

  byteFlag: 0x7E,
  byteControlEscape: 0x7D,

  MESSAGE_ID_HEARTBEAT: 0,
  MESSAGE_ID_INITIALIZATION: 2,
  MESSAGE_ID_UPLINK: 7,
  MESSAGE_ID_HAT: 9,
  MESSAGE_ID_OWNSHIP: 10,
  MESSAGE_ID_OWNSHIP_GEO_ALT: 11,
  MESSAGE_ID_TRAFFIC: 20,
  MESSAGE_ID_BASIC: 30,
  MESSAGE_ID_LONG: 31,
  MESSAGE_ID_1090_SHORT: 32,
  MESSAGE_ID_1090_LONG: 33,
  MESSAGE_ID_UAVIONIX: 117,

  UAVIONIX_SIGNATURE: 65,

  UAVIONIX_OEM_TRANSCEIVER_CONFIG: 1,
  UAVIONIX_OEM_TRANSCEIVER_SETUP: 2,
  UAVIONIX_OEM_CONFIGURATION_REPORT: 3,
  UAVIONIX_OEM_SYSCMD_SET_BAUD_RATE: 4,
  UAVIONIX_OEM_SYSCMD_SET_LED_BRIGHTNESS: 5,
  UAVIONIX_OEM_SYSCMD_ADJUST_FILTERS: 6,
  UAVIONIX_OEM_SYSCMD_ENTER_GDL90_PLUS: 7,
  UAVIONIX_OEM_STATUS_REPORT: 8,
  UAVIONIX_OEM_TOWER_STATUS_REPORT: 9,
  UAVIONIX_OEM_CONTROL_PANEL: 10,
  UAVIONIX_OEM_DEVICE_STARTUP: 11,
  UAVIONIX_OEM_SOLICIT_RESPONSE: 253,
  UAVIONIX_OEM_SYSCMD_ENTER_UPDATE: 254,

  UA_SUBTYPE_STATIC: 0x01,
  UA_SUBTYPE_SETUP: 0x02,
  UA_SUBTYPE_STATUS: 0x03,
  UA_SUBTYPE_BAUDRATE: 0x04,
  UA_SUBTYPE_UPDATE_MODE: 0x05,
  UA_SUBTYPE_GEOFENCE: 0x25,

  GEOFENCE_SUBTYPE_NONE: 0,
  GEOFENCE_SUBTYPE_RESET: 1,
  GEOFENCE_SUBTYPE_POINTS: 2,
  GEOFENCE_SUBTYPE_POLYGONS: 3,
  GEOFENCE_SUBTYPE_VERIFY: 4,
  GEOFENCE_SUBTYPE_ACK: 0x80,

  GEOFENCE_ERROR_NONE: 0,
  GEOFENCE_ERROR_UNKNOWN: -1,
  GEOFENCE_ERROR_TOO_MANY_POINTS: -2,
  GEOFENCE_ERROR_TOO_MANY_POLYGONS: -3,
  GEOFENCE_ERROR_CRC_MISMATCHED: -4,
  GEOFENCE_ERROR_SEQUENCE: -5,

  EMIT_NONE: 0,
  EMIT_LIGHT: 1,
  EMIT_SMALL: 2,
  EMIT_LARGE: 3,
  EMIT_HIGH_VORTEX_LARGE: 4,
  EMIT_HEAVY: 5,
  EMIT_HIGHTLY_MANUEVERABLE: 6,
  EMIT_ROTORCRAFT: 7,
  EMIT_GLIDER: 9,
  EMIT_LIGHTER_THAN_AIR: 10,
  EMIT_PARACHUTE: 11,
  EMIT_ULTRALIGHT: 12,
  EMIT_UAV: 14,
  EMIT_SPACE: 15,
  EMIT_SURFACE_VEHICLE_EMERGENCY: 17,
  EMIT_SURFACE_VEHICLE_SERVICE: 18,
  EMIT_POINT_OBSTACLE: 19,
  EMIT_CLUSTER_OBSTACLE: 20,
  EMIT_LINE_OBSTACLE: 21,

  emitterTypes: ["None", "Light", "Small", "Large", "High Vortex Large", "Heavy", "Highly Maneuverable", "Rotorcraft", "Unassigned", "Glider/Sailplane", "Ligher Than Air", "Parachutist/Sky Diver", "Ultralight/Hang Glider/Paraglider", "Unassigned", "Unmanned Aerial Vehicle", "Space", "Unassigned", "Surface Vehicle - Emergency", "Surface Vehicle - Service", "Point Obstacle", "Cluster Obstacle", "Line Obstacle"],

  EMERG_NONE: 0,
  EMERG_GENERAL: 1,
  EMERG_MEDICAL: 2,
  EMERG_MIN_FUEL: 3,
  EMERG_NO_COMM: 4,
  EMERG_UNLAWFUL_INTERFERENCE: 5,
  EMERG_DOWNED: 6,

  emergencyTypes: ["None", "General", "Medical", "Minimum Fuel", "No Communication", "Unlawful Interference", "Downed Aircraft"],

  // GDL90 Constructor
  initialize: function() {
    console.log("GDL90 intialized");
  },
  // Search packet for a byte
  // Similar to indexOf, but cross-browser
  // start is optional index to start searching
  // Returns index of first match
  packetSearch: function(packet, search, start) {
    start = (typeof start !== 'undefined') ? start : 0;

    var index;
    for (index=start; index < packet.length; index++) {
      if (packet[index] == search) return index;
    }
    return -1;
  },
  // Unframe and FCS check packets
  // Input is typedarray
  // Returns array of objects with .valid, .type, and .packet
  // .packet is unstuffed and has flag bytes and FCS removed
  validatePackets: function(packet) {

    var packetArray = new Array();

    // Minimum packet size is 5 bytes
    while (packet.length >= 5) {

      var packetValid = false;
      var packetType = 0x00;
      var packetLength = 0;
      var unstuffedPacket = null;

      // Validate flag bytes
      if (packet[0] == this.byteFlag) {
        packet = packet.subarray(1);
        // Find end flag byte
        packetLength = this.packetSearch(packet, this.byteFlag);
        if (packetLength > 0) {
          // Unstuff, discard flag bytes
          var unstuffed = [];
          for (var i=0; i < packetLength; i++) {
            if (packet[i] == this.byteControlEscape) {
              i++;
              unstuffed.push(packet[i] | 0x20);
            } else {
              unstuffed.push(packet[i]);
            }
          }
          var fcsReceived = unstuffed[unstuffed.length-1] << 8;
          fcsReceived = fcsReceived | unstuffed[unstuffed.length-2];

          // Extract data - no flag bytes or FCS
          unstuffedPacket = new Uint8Array(unstuffed.slice(0, unstuffed.length-2));
          // Calculate FCS over message ID and data
          var fcsCalculated = crc16.calculate(unstuffedPacket);

          if (fcsReceived == fcsCalculated) {
            packetValid = true;
            packetType = unstuffed[0];
          }
          packet = packet.subarray(packetLength+1);
        } else {
          // We couldn't find an end flag, we're done
          packet = new Array();
        }

        packetArray.push({valid: packetValid, type: packetType, packet: unstuffedPacket});
      }
      else
      {
        // Dump entire packet. We'd be more compatible if we scanned through the bytes looking for a flag. 
        packet = new Array();
      }
        
    }
    return packetArray;
  },
  // Perform byte stuffing
  // Input is a typedarray
  byteStuff: function(packet) {
     // Look for flag and control-escape bytes
     // Don't bytestuff the bookmarking flag bytes
     var byteStuffed = [];
     var readIndex;
     var writeIndex = 0;
     for (readIndex=0; readIndex < packet.length; readIndex++) {
       if ((readIndex == 0) || (readIndex == (packet.length-1))) {
         byteStuffed[writeIndex++] = packet[readIndex];
       } else if (packet[readIndex] == this.byteFlag || packet[readIndex] == this.byteControlEscape) {
         byteStuffed[writeIndex++] = this.byteControlEscape;
         byteStuffed[writeIndex++] = packet[readIndex] ^ 0x20;
       } else {
         byteStuffed[writeIndex++] = packet[readIndex];
       }
     }
     return new Uint8Array(byteStuffed);
  },
  // Marshall the uAvionix GDL90 Static packet
  marshallStatic: function() {
    var packet = new Uint8Array(19);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = 0x75;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - static config
    packet[packetIndex++] = 0x01;
    // Version - 1
    packet[packetIndex++] = 0x01;
    // ICAO and Emitter Type
    packet[packetIndex++] = (parseInt(document.getElementById("icao").value, 16) >> 16) & 0xFF;
    packet[packetIndex++] = (parseInt(document.getElementById("icao").value, 16) >> 8) & 0xFF;
    packet[packetIndex++] = parseInt(document.getElementById("icao").value, 16) & 0xFF;
    packet[packetIndex] = packet[packetIndex] | (0xFF & document.getElementById("emitter").value);
    packetIndex++;
    // Callsign
    for (var i=0; i<8; i++) {
      // Pad with spaces
      var callsignCharacter = document.getElementById("callsign").value.charAt(i);
      if (callsignCharacter == "") {
        callsignCharacter = " ";
      }
      packet[packetIndex++] = callsignCharacter.charCodeAt(0);
    }
    if (document.getElementById("Vs0Span").style.display != "none") {
      packet[packetIndex++] = parseInt(document.getElementById("Vs0").value, 10);
    } else {
      packet[packetIndex++] = 0xFF;
    }
    // Aircraft length and width, bits 2-5 as in table 2-34
    // If length is 8 that means >85 meters, which is actually 7
    var aLength = parseInt(document.getElementById("aLength").value, 10);
    if (aLength == 8) {
      packet[packetIndex] = 7 << 1;
    } else {
      packet[packetIndex] = aLength << 1;
    }
    var aWidth = parseInt(document.getElementById("aWidth").value.charAt(2), 10);
    packet[packetIndex] = (packet[packetIndex] | (0x01 & aWidth));
    packetIndex++;
    // Antenna offset, tables 2-36 and 2-37, lateral (3 bits) then longitudinal (5 bits)
    packet[packetIndex] = parseInt(document.getElementById("offsetLat").value, 10) << 5;
    packet[packetIndex] = packet[packetIndex] | (0x1F & parseInt(document.getElementById("offsetLon").value, 10));
    packetIndex++;
    return packet;
  },
  // Marshall and construct a uAvionix GDL 90 Static packet (0x75) with FCS
  constructStatic: function() {
     // Length is 19 bytes plus 2 flags and FCS (23 bytes)
     var packet = new Uint8Array(23);
     packet[0] = this.byteFlag;
     packet.set(this.marshallStatic(), 1);
     var fcs = crc16.calculate(packet.subarray(1, 20));
     packet[20] = fcs & 0xFF;
     packet[21] = (fcs >> 8) & 0xFF;
     packet[22] = this.byteFlag;
     // Bytestuff
     return this.byteStuff(packet);
  },
  // Marshall the uAvionix GDL90 Setup packet
  marshallSetupV1: function(sil, sda, threshold) {
    var packet = new Uint8Array(8);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = 0x75;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - static setup
    packet[packetIndex++] = 0x02;
    // Version - 1
    packet[packetIndex++] = 0x01;
    // SIL
    packet[packetIndex++] = sil;
    // SDA
    packet[packetIndex++] = sda;
    // Threshold (uint16_t)
    packet[packetIndex++] = threshold & 0xFF;
    packet[packetIndex++] = (threshold >> 8) & 0xFF;
    return packet;
  },
  // Marshall and construct a uAvionix GDL 90 Static Setup packet with FCS
  constructSetupV1: function(sil, sda, threshold) {
    // Length is 8 bytes plus 2 flags and FCS (12 bytes)
    var packet = new Uint8Array(12);
    packet[0] = this.byteFlag;
    packet.set(this.marshallSetupV1(sil, sda, threshold), 1);
    var fcs = crc16.calculate(packet.subarray(1, 9));
    packet[9] = fcs & 0xFF;
    packet[10] = (fcs >> 8) & 0xFF;
    packet[11] = this.byteFlag;
    // Bytestuff
    return this.byteStuff(packet);
  },
  // Marshall the uAvionix GDL90 Setup packet (v2)
  marshallSetupV2: function(sil, sda, threshold, outputFormat, control) {
    var packet = new Uint8Array(10);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = 0x75;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - static setup
    packet[packetIndex++] = 0x02;
    // Version - 2
    packet[packetIndex++] = 0x02;
    // SIL
    packet[packetIndex++] = sil;
    // SDA
    packet[packetIndex++] = sda;
    // Threshold (uint16_t)
    packet[packetIndex++] = threshold & 0xFF;
    packet[packetIndex++] = (threshold >> 8) & 0xFF;
    // Output format
    packet[packetIndex++] = outputFormat;
    // Control
    packet[packetIndex++] = control;
    return packet;
  },
  // Marshall and construct a uAvionix GDL 90 Static Setup V2 packet with FCS
  constructSetupV2: function(sil, sda, threshold, outputFormat, control) {
    // Length is 10 bytes plus 2 flags and FCS (14 bytes)
    var packet = new Uint8Array(14);
    packet[0] = this.byteFlag;
    packet.set(this.marshallSetupV2(sil, sda, threshold, outputFormat, control), 1);
    var fcs = crc16.calculate(packet.subarray(1, 11));
    packet[11] = fcs & 0xFF;
    packet[12] = (fcs >> 8) & 0xFF;
    packet[13] = this.byteFlag;
    // Bytestuff
    return this.byteStuff(packet);
  },
  // Marshall the uAvionix GDL90 Setup packet (v3)
  marshallSetupV3: function(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn) {
    var packet = new Uint8Array(14);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = 0x75;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - static setup
    packet[packetIndex++] = 0x02;
    // Version - 3
    packet[packetIndex++] = 0x03;
    // SIL
    packet[packetIndex++] = sil;
    // SDA
    packet[packetIndex++] = sda;
    // Threshold (uint16_t)
    packet[packetIndex++] = threshold & 0xFF;
    packet[packetIndex++] = (threshold >> 8) & 0xFF;
    // Output format
    packet[packetIndex++] = outputFormat;
    // Control
    packet[packetIndex++] = control;
    // Default VFR Code
    packet[packetIndex++] = defaultVfrCode & 0xFF;
    packet[packetIndex++] = (defaultVfrCode >> 8) & 0xFF;
    // Max aircraft speed
    packet[packetIndex++] = maxSpeed;
    // ADS-B In capability
    packet[packetIndex++] = adsbIn;
    return packet;
  },
  // Marshall and construct a uAvionix GDL 90 Static Setup V3 packet with FCS
  constructSetupV3: function(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn) {
    // Length is 14 bytes plus 2 flags and FCS (18 bytes)
    var packet = new Uint8Array(18);
    packet[0] = this.byteFlag;
    packet.set(this.marshallSetupV3(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn), 1);
    var fcs = crc16.calculate(packet.subarray(1, 15));
    packet[15] = fcs & 0xFF;
    packet[16] = (fcs >> 8) & 0xFF;
    packet[17] = this.byteFlag;
    // Bytestuff
    return this.byteStuff(packet);
  },
  // Marshall the uAvionix GDL90 Transceiver Setup packet (v4)
  marshallSetupV4: function(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous, flightPlanId, setupSource, controlSources, gpsSource, displayOutput, com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol) {
    var packet = new Uint8Array(42);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = 0x75;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - static setup
    packet[packetIndex++] = 0x02;
    // Version - 4
    packet[packetIndex++] = 0x04;
    // SIL
    packet[packetIndex++] = sil;
    // SDA
    packet[packetIndex++] = sda;
    // Threshold (uint16_t)
    packet[packetIndex++] = threshold & 0xFF;
    packet[packetIndex++] = (threshold >> 8) & 0xFF;
    // Output format
    packet[packetIndex++] = outputFormat;
    // Control
    packet[packetIndex++] = control;
    // Default VFR Code
    packet[packetIndex++] = defaultVfrCode & 0xFF;
    packet[packetIndex++] = (defaultVfrCode >> 8) & 0xFF;
    // Max aircraft speed
    packet[packetIndex++] = maxSpeed;
    // ADS-B In capability
    packet[packetIndex++] = adsbIn;
    // CSID Logic
    packet[packetIndex++] = csidLogic;
    // Anonymous mode
    packet[packetIndex++] = anonymous;
    // Flight Plan ID
    for (var i=0; i<8; i++) {
      if (flightPlanId != null) {
        // Pad with spaces
        var flightPlanCharacter = flightPlanId.charAt(i);
        if (flightPlanCharacter == "") {
          flightPlanCharacter = " ";
        }
        packet[packetIndex++] = flightPlanCharacter.charCodeAt(0);
      } else {
        packet[packetIndex++] = 0xFF;
      }
    }
    packet[packetIndex++] = setupSource;
    packet[packetIndex++] = controlSources;
    packet[packetIndex++] = gpsSource;
    packet[packetIndex++] = displayOutput;
    packet[packetIndex++] = com1Rate;
    packet[packetIndex++] = com1Data;
    packet[packetIndex++] = com1PHY;
    packet[packetIndex++] = com1InputProtocol & 0xFF;
    packet[packetIndex++] = (com1InputProtocol >> 8) & 0xFF;
    packet[packetIndex++] = com1OutputProtocol & 0xFF;
    packet[packetIndex++] = (com1OutputProtocol >> 8) & 0xFF;
    packet[packetIndex++] = com2Rate;
    packet[packetIndex++] = com2Data;
    packet[packetIndex++] = com2PHY;
    packet[packetIndex++] = com2InputProtocol & 0xFF;
    packet[packetIndex++] = (com2InputProtocol >> 8) & 0xFF;
    packet[packetIndex++] = com2OutputProtocol & 0xFF;
    packet[packetIndex++] = (com2OutputProtocol >> 8) & 0xFF;

    return packet;
  },
  // Marshall and construct a uAvionix GDL 90 Transceiver Setup V4 packet with FCS
  constructSetupV4: function(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous, flightPlanId, setupSource, controlSources, gpsSource, displayOutput, com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol) {
    // Length is 42 bytes plus 2 flags and FCS (46 bytes)
    var packet = new Uint8Array(46);
    packet[0] = this.byteFlag;
    packet.set(this.marshallSetupV4(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous, flightPlanId, setupSource, controlSources, gpsSource, displayOutput, com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol), 1);
    var fcs = crc16.calculate(packet.subarray(1, 43));
    packet[43] = fcs & 0xFF;
    packet[44] = (fcs >> 8) & 0xFF;
    packet[45] = this.byteFlag;
    // Bytestuff
    return this.byteStuff(packet);
  },
  // Marshall the uAvionix GDL90 Transceiver Setup packet (v5)
  marshallSetupV5: function(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous, flightPlanId, setupSource, controlSources, gpsSource, displayOutput, com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol, baroAltSource) {
    var packet = new Uint8Array(43);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = 0x75;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - static setup
    packet[packetIndex++] = 0x02;
    // Version - 5
    packet[packetIndex++] = 0x05;
    // SIL
    packet[packetIndex++] = sil;
    // SDA
    packet[packetIndex++] = sda;
    // Threshold (uint16_t)
    packet[packetIndex++] = threshold & 0xFF;
    packet[packetIndex++] = (threshold >> 8) & 0xFF;
    // Output format
    packet[packetIndex++] = outputFormat;
    // Control
    packet[packetIndex++] = control;
    // Default VFR Code
    packet[packetIndex++] = defaultVfrCode & 0xFF;
    packet[packetIndex++] = (defaultVfrCode >> 8) & 0xFF;
    // Max aircraft speed
    packet[packetIndex++] = maxSpeed;
    // ADS-B In capability
    packet[packetIndex++] = adsbIn;
    // CSID Logic
    packet[packetIndex++] = csidLogic;
    // Anonymous mode
    packet[packetIndex++] = anonymous;
    // Flight Plan ID
    for (var i=0; i<8; i++) {
      if (flightPlanId != null) {
        // Pad with spaces
        var flightPlanCharacter = flightPlanId.charAt(i);
        if (flightPlanCharacter == "") {
          flightPlanCharacter = " ";
        }
        packet[packetIndex++] = flightPlanCharacter.charCodeAt(0);
      } else {
        packet[packetIndex++] = 0xFF;
      }
    }
    packet[packetIndex++] = setupSource;
    packet[packetIndex++] = controlSources;
    packet[packetIndex++] = gpsSource;
    packet[packetIndex++] = displayOutput;
    packet[packetIndex++] = com1Rate;
    packet[packetIndex++] = com1Data;
    packet[packetIndex++] = com1PHY;
    packet[packetIndex++] = com1InputProtocol & 0xFF;
    packet[packetIndex++] = (com1InputProtocol >> 8) & 0xFF;
    packet[packetIndex++] = com1OutputProtocol & 0xFF;
    packet[packetIndex++] = (com1OutputProtocol >> 8) & 0xFF;
    packet[packetIndex++] = com2Rate;
    packet[packetIndex++] = com2Data;
    packet[packetIndex++] = com2PHY;
    packet[packetIndex++] = com2InputProtocol & 0xFF;
    packet[packetIndex++] = (com2InputProtocol >> 8) & 0xFF;
    packet[packetIndex++] = com2OutputProtocol & 0xFF;
    packet[packetIndex++] = (com2OutputProtocol >> 8) & 0xFF;
    packet[packetIndex++] = baroAltSource;

    return packet;
  },
  // Marshall and construct a uAvionix GDL 90 Transceiver Setup V5 packet with FCS
  constructSetupV5: function(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous, flightPlanId, setupSource, controlSources, gpsSource, displayOutput, com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol, baroAltSource) {
    // Length is 43 bytes plus 2 flags and FCS (42 bytes)
    var packet = new Uint8Array(47);
    packet[0] = this.byteFlag;
    packet.set(this.marshallSetupV5(sil, sda, threshold, outputFormat, control, defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous, flightPlanId, setupSource, controlSources, gpsSource, displayOutput, com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol, baroAltSource), 1);
    var fcs = crc16.calculate(packet.subarray(1, 44));
    packet[44] = fcs & 0xFF;
    packet[45] = (fcs >> 8) & 0xFF;
    packet[46] = this.byteFlag;
    // Bytestuff
    return this.byteStuff(packet);
  },
  marshallGeofence: function(geofenceMsgType, sequence) {
    var packet = new Uint8Array(6);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = this.MESSAGE_ID_UAVIONIX;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - geofence
    packet[packetIndex++] = 0x25;
    // Version - 1
    packet[packetIndex++] = 0x01;
	// Sequence
    packet[packetIndex++] = sequence;
    // Geofence subtype
    packet[packetIndex++] = geofenceMsgType;
    return packet;
  },
  constructGeofence: function(geofenceMsgType, sequence, geofencePayload) {
    var packetHeader = this.marshallGeofence(geofenceMsgType, sequence);
    // +4 for byteFlags and CRC
    var packet = new Uint8Array(packetHeader.length + geofencePayload.length + 4);
    packet[0] = this.byteFlag;
    packet.set(packetHeader, 1);
    packet.set(geofencePayload, 1 + packetHeader.length);
    var fcs = crc16.calculate(packet.subarray(1, packet.length - 3));
    packet[packet.length - 3] = fcs & 0xFF;
    packet[packet.length - 2] = (fcs >> 8) & 0xFF;
    packet[packet.length - 1] = this.byteFlag;
    return this.byteStuff(packet);
  },
  // Marshall the uAvionix Solicit Report packet
  marshallSolicitReport: function(messageId, subtype) {
    var packet = new Uint8Array(6);
    var packetIndex = 0;
    // Type
    packet[packetIndex++] = 0x75;
    // Signature
    packet[packetIndex++] = 0x41;
    // Subtype - solicit response
    packet[packetIndex++] = 0xFD;
    // Version - 1
    packet[packetIndex++] = 0x01;
    // Solicited message ID
    packet[packetIndex++] = messageId;
    // Solicited subtype
    packet[packetIndex++] = subtype;
    return packet;
  },
  // Marshall and construct a uAvionix GDL90 Solicit Status Report packet with FCS
  constructSolicitStatusReport: function() {
    // Length is 6 bytes plus 2 flags and FCS (10 bytes)
    var packet = new Uint8Array(10);
    packet[0] = this.byteFlag;
    // Marshall a solicit report for a Status Report (0x75/0x08)
    packet.set(this.marshallSolicitReport(0x75, 0x08), 1);
    var fcs = crc16.calculate(packet.subarray(1, 7));
    packet[7] = fcs & 0xFF;
    packet[8] = (fcs >> 8) & 0xFF;
    packet[9] = this.byteFlag;
    // Bytestuff
    return this.byteStuff(packet);
  },
  toHexString: function(byteArray) {
    var hexString = "";
    for (var i=0; i < byteArray.length; i++) {
      var hexDigit = byteArray[i].toString(16);
      hexString += hexDigit.length > 1 ? hexDigit : ('0' + hexDigit);
    }
    return hexString;  
  },
  toUint8: function(byteArray) {
    return byteArray[0];
  },
  // We need to byte swap - data comes across as MSB first
  toUint16: function(byteArray) {
    var u8a = new Uint8Array(2);
    u8a.set(byteArray.subarray(1,2), 0);
    u8a.set(byteArray.subarray(0,1), 1);
    return new Uint16Array(u8a.buffer, 0, 1)[0];
  },
  // Can accept arrays of less than 4 bytes
  // We need to byte swap - data comes across as MSB first
  toUint32: function(byteArray) {
    var u8a = new Uint8Array(4);
    var arrayLen = Math.min(4, byteArray.length);
    for (var i=0; i < arrayLen; i++) {
      u8a.set(byteArray.subarray(arrayLen-i-1, arrayLen-i), i);  
    }
    return new Uint32Array(u8a.buffer, 0, 1)[0];
  },
  decodeLatLon: function(value) {
    var deg = value;
    if (value & 0x800000) {
      deg = (value & 0x7fffff)-0x800000;
    }
    deg = deg * 180 / 0x800000;
    return deg;
  },
  decodeOwnship: function(packet) {
    var ownship = new Object;
    ownship.raw = new Object;

    ownship.raw.address = packet.subarray(2,5);
    ownship.address = this.toHexString(ownship.raw.address).toUpperCase();

    ownship.raw.callsign = packet.subarray(19,27);
    ownship.callsign= "";
    for (var i=0; i < ownship.raw.callsign.length; i++) {
      ownship.callsign += String.fromCharCode(ownship.raw.callsign[i]);
    }

    // Lat/long are 24-bit signed binary fractions
    ownship.raw.latitude = packet.subarray(5,8);
    ownship.raw.longitude = packet.subarray(8,11);
    // decode lat/lon
    ownship.latitude = this.decodeLatLon(this.toUint32(ownship.raw.latitude));
    ownship.longitude = this.decodeLatLon(this.toUint32(ownship.raw.longitude));
    // altitude in feet = ddd * 25 - 1000
    // If unavailable, 0xfff
    ownship.raw.altitude = packet.subarray(11,13);
    ownship.altitude = this.toUint16(ownship.raw.altitude) >>> 4;
    if (ownship.altitude == 0xfff) {
      ownship.altitude = "--";
    } else {
      ownship.altitude = ownship.altitude * 25 - 1000;
    }

    // Misc
    ownship.raw.misc = packet.subarray(12,13);
    ownship.misc = this.toUint8(ownship.raw.misc) & 0x0f;

    // NIC / NACp
    ownship.raw.nicNacp = packet.subarray(13,14);
    ownship.NIC = this.toUint8(ownship.raw.nicNacp) >>> 4;
    ownship.NACp = this.toUint8(ownship.raw.nicNacp) & 0x0f;

    // If lat=lon=NIC=0, then position unavailable
    if (ownship.latitude == 0 && ownship.longitude == 0 && ownship.NIC == 0) {
      ownship.latitude = "--";
      ownship.longitude = "--";
    }

    // Horizontal velocity, 12-bit unsigned in knots
    // If unavailable, 0xfff
    ownship.raw.velocityHorizontal = packet.subarray(14,16);
    ownship.velocityHorizontal = this.toUint16(ownship.raw.velocityHorizontal) >>> 4;
    if (ownship.velocityHorizontal == 0xfff) {
      ownship.velocityHorizontal = "--";
    }
    // Vertical velocity, 12-bit signed in units of 64 FPM
    // 0x000 is 0, 0x001 is +64, 0xFFF is -64, 0x800 is unavailable
    // If unavailable, 0x800
    ownship.raw.velocityVertical = packet.subarray(15,17);
    ownship.velocityVertical = this.toUint16(ownship.raw.velocityVertical) & 0xfff;
    if (ownship.velocityVertical == 0x800) {
      ownship.velocityVertical = "--";
    } else {
      if (ownship.velocityVertical & 0x800) {
        // Negative value, convert
        ownship.velocityVertical = (ownship.velocityVertical & 0x7ff) - 0x800; 
      }
      ownship.velocityVertical = ownship.velocityVertical * 64;
    }
    // Track/heading
    // 8-bit angular weighted, units of 360/256 degrees
    // RCB check misc bits to see if TT valid
    ownship.raw.trackHeading = packet.subarray(17,18);
    ownship.trackHeading = this.toUint8(ownship.raw.trackHeading);
    ownship.trackHeading = ownship.trackHeading * 360 / 256;

    // If TT isn't valid
    if ((ownship.misc & 0x03) == 0x00) {
      ownship.trackHeading = "--";
    } 
    
    // Emitter category
    ownship.raw.emitterCategory = packet.subarray(18,19);
    ownship.emitterCategory = this.toUint8(ownship.raw.emitterCategory);
    ownship.emitterCategoryName = this.emitterTypes[ownship.emitterCategory];
    // Emergency
    ownship.raw.emergency = packet.subarray(27,28);
    ownship.emergency = this.toUint8(ownship.raw.emergency) >>> 4;
    ownship.emergencyName = this.emergencyTypes[ownship.emergency];
    return ownship;
  },
  decodeOwnshipGeoAlt: function(packet) {
    var ownshipGeoAlt = new Object;
    ownshipGeoAlt.raw = new Object;

    // 16-bit signed, 5-foot units
    ownshipGeoAlt.raw.altitude = packet.subarray(1,3);
    ownshipGeoAlt.altitude = this.toUint16(ownshipGeoAlt.raw.altitude);
    if (ownshipGeoAlt.altitude & 0x8000) {
      // Negative value, convert
      ownshipGeoAlt.altitude = (ownshipGeoAlt.altitude & 0x7fff) - 0x8000; 
    }
    ownshipGeoAlt.altitude = ownshipGeoAlt.altitude * 5;

    ownshipGeoAlt.raw.metrics = packet.subarray(3,5);
    // RCB TODO decode vertical metrics
    return ownshipGeoAlt;
  },
  decodeConfigurationReport: function(packet) {
    var configReport = new Object;
    configReport.raw = new Object;

    configReport.gnssDeviceType = packet[4];
    configReport.adsbFwId = packet[5];
    configReport.adsbHwId = packet[6];
    configReport.adsbHwVersion = packet[7];
    configReport.adsbSwVersionMajor = packet[8];
    configReport.adsbSwVersionMinor = packet[9];
    configReport.adsbSwVersionRevision = packet[10];
    configReport.gnssHwVersion = packet[11];
    configReport.gnssSwVersionMajor = packet[12];
    configReport.gnssSwVersionMinor = packet[13];
    configReport.gnssSwVersionRevision = packet[14];

    configReport.raw.stallSpeed = packet.subarray(15,17);
    configReport.stallSpeed = this.toUint16(configReport.raw.stallSpeed);
    configReport.raw.avLw = packet[17];
    // Convert referencing DO-282B Table 2-35
    // aWidth is stored as LengthCode.WidthCode, e.g. 7.1 to match the selection
    // We will never "restore" alength of >85m or width >90m as they are not unique,
    // they overlap with the top bounded values
    configReport.aLength = configReport.raw.avLw >> 1;
    configReport.aWidth = String(configReport.aLength);
    configReport.aWidth += (configReport.raw.avLw & 0x01) ? ".1" : ".0";
    configReport.raw.antOffsetLatLon = packet[18];
    // RCB convert avLw and antOffsetLatLon
    configReport.avLw = configReport.raw.avLw;
    configReport.antOffsetLat = configReport.raw.antOffsetLatLon >> 5;
    configReport.antOffsetLon = configReport.raw.antOffsetLatLon & 0x1F;
    configReport.sil = packet[19];
    configReport.sda = packet[20];
    configReport.outputFormat = packet[21];
    configReport.txControl = packet[22];
    configReport.raw.sniffThreshold = packet.subarray(23,25);
    configReport.sniffThreshold = this.toUint16(configReport.raw.sniffThreshold);
    configReport.raw.defaultVfrCode = packet.subarray(25,27);
    configReport.defaultVfrCode = this.toUint16(configReport.raw.defaultVfrCode);
    configReport.maxSpeed = packet[27];
    configReport.adsbIn = packet[28];

    return configReport;
  },
  decodeConfigurationReportV2: function(packet) {

    // decode V1 fields
    var configReport = this.decodeConfigurationReport(packet);

    configReport.csidLogic = packet[29];
    configReport.anonymous = packet[30];
    configReport.raw.flightPlanId = packet.subarray(31,39);
    configReport.flightPlanId = "";
    for (var i=0; i < configReport.raw.flightPlanId.length; i++) {
      configReport.flightPlanId+= String.fromCharCode(configReport.raw.flightPlanId[i]);
    }
    configReport.setupSource = packet[39];
    configReport.controlSources = packet[40];
    configReport.gpsSource = packet[41];
    configReport.displayOutput = packet[42];
    configReport.com1Rate = packet[43];
    configReport.com1Data = packet[44];
    configReport.com1PHY = packet[45];
    configReport.raw.com1InputProtocol = packet.subarray(46,48);
    configReport.com1InputProtocol = this.toUint16(configReport.raw.com1InputProtocol);
    configReport.raw.com1OutputProtocol = packet.subarray(48,50);
    configReport.com1OutputProtocol = this.toUint16(configReport.raw.com1OutputProtocol);
    configReport.com2Rate = packet[50];
    configReport.com2Data = packet[51];
    configReport.com2PHY = packet[52];
    configReport.raw.com2InputProtocol = packet.subarray(53,55);
    configReport.com2InputProtocol = this.toUint16(configReport.raw.com2InputProtocol);
    configReport.raw.com2OutputProtocol = packet.subarray(55,57);
    configReport.com2OutputProtocol = this.toUint16(configReport.raw.com2OutputProtocol);

    configReport.messageVersion = 2;

    return configReport;
  },
  decodeConfigurationReportV3: function(packet) {

    // decode V2 fields
    var configReport = this.decodeConfigurationReportV2(packet);

    configReport.raw.callsign = packet.subarray(57,65);
    configReport.callsign = "";
    for (var i=0; i < configReport.raw.callsign.length; i++) {
      configReport.callsign += String.fromCharCode(configReport.raw.callsign[i]);
    }
    configReport.raw.address = packet.subarray(65,68);
    configReport.address = this.toHexString(configReport.raw.address).toUpperCase();
    configReport.raw.uuid = packet.subarray(68,76);
    configReport.uuid = this.toHexString(configReport.raw.uuid).toUpperCase();
    configReport.baroAltSource = packet[76];

    configReport.messageVersion = 3;

    return configReport;
  },
  decodeConfigurationReportV4: function(packet) {

    // decode V3 fields
    var configReport = this.decodeConfigurationReportV3(packet);

    configReport.ledCapabilities = packet[77];
    configReport.ledState = packet[78];

    configReport.messageVersion = 4;

    return configReport;
  },
  decodeConfigurationReportV5: function(packet) {

    // decode V4 fields
    var configReport = this.decodeConfigurationReportV4(packet);

    configReport.emitterCategory = packet[79];
    configReport.emitterCategoryName = this.emitterTypes[configReport.emitterCategory];
    
    configReport.messageVersion = 5;

    return configReport;
  },
  decodeConfigurationReportV6: function(packet) {

    // decode V5 fields
    var configReport = this.decodeConfigurationReportV5(packet);

    configReport.tmFlightPlanIdTimeout = packet[80];
    configReport.tmPressureAltitudeTimeout = packet[81];
    configReport.raw.tmPressureAltitudeTolerance = packet.subarray(82,84);
    configReport.tmPressureAltitudeTolerance = this.toUint16(configReport.raw.tmPressureAltitudeTolerance);
    configReport.tmPressureAltitudeFilterWidth = packet[84];
    configReport.pressureAltitudeMode = packet[85];
    configReport.raw.pressureAltitudeCalibrationOffset = packet.subarray(86,90);
    configReport.pressureAltitudeCalibrationOffset = this.toUint32(configReport.raw.pressureAltitudeCalibrationOffset);

    configReport.raw.crcAdsb = packet.subarray(90,94);
    configReport.crcAdsb = this.toHexString(configReport.raw.crcAdsb).toUpperCase();
    configReport.raw.crcGnss = packet.subarray(94,98);
    configReport.crcGnss = this.toHexString(configReport.raw.crcGnss).toUpperCase();   

    configReport.messageVersion = 6;

    return configReport;    
  },
  
  decodeStatusReport: function(packet) {
    var statusReport = new Object;
    statusReport.raw = new Object;

    statusReport.transceiverStatus = packet[4];
    statusReport.transceiverCapabilities = packet[5];
    statusReport.gpsSvsInUse = packet[6];
    statusReport.gpsSvsInView = packet[7];
    statusReport.gpsFixQuality = packet[8];
    statusReport.numGroundTowers = packet[9];
    statusReport.raw.numUatTargets = packet.subarray(10,12);
    statusReport.numUatTargets = this.toUint16(statusReport.raw.numUatTargets);
    statusReport.raw.num1090Targets = packet.subarray(12,14);
    statusReport.num1090Targets = this.toUint16(statusReport.raw.num1090Targets);
    statusReport.raw.trafficReportsPerSecond = packet.subarray(14,16);
    statusReport.trafficReportsPerSecond = this.toUint16(statusReport.raw.trafficReportsPerSecond);
    statusReport.raw.receiveFrameCount1090 = packet.subarray(16,18);
    statusReport.receiveFrameCount1090 = this.toUint16(statusReport.raw.receiveFrameCount1090);

    // RCB TODO complete decode of status report

    statusReport.raw.currentFlightplan = packet.subarray(36,44);
    statusReport.currentFlightplan = "";
    for (var i=0; i < statusReport.raw.currentFlightplan.length; i++) {
      statusReport.currentFlightplan += String.fromCharCode(statusReport.raw.currentFlightplan[i]);
    }

    statusReport.messageVersion = 1;

    return statusReport;
  },
  decodeStatusReportV2: function(packet) {
    // decode V1 fields
    var statusReport = this.decodeStatusReport(packet);

    // RCB complete decode of status report

    statusReport.messageVersion = 2;

    return statusReport;
  }
};
